#!/bin/sh

#lumens dante path
export DEP_EXE_PATH=/usr/local/dante/dep/dante_package

CONTAINER_RUNTIME=crun
CONTAINER_STATUS_PATH=/run/$CONTAINER_RUNTIME
CONTAINER_CMD="${DEP_EXE_PATH}/${CONTAINER_RUNTIME} --root=${CONTAINER_STATUS_PATH}"

IMAGES_PATH=$DEP_EXE_PATH/dante_data/images
ROOTFS_MOUNTPOINT=$DEP_EXE_PATH/bundle/rootfs
ACTIVE_IMAGE_ID_PATH=$IMAGES_PATH/active

# Check we can actually start/stop containers
if [ "$(id -u)" -ne 0 ]; then
	echo "This script must be executed with root privileges."
	echo ""
	exit 1
fi

start()
{
	# A poorly-timed stop() could leave the container mounted while
	# the processes inside the container were successfully shut down.
	# Instead of relying on whether the container is there or not when 
	# deciding to start DEP, check whether dep_manager is actually running.

	# shellcheck disable=SC2009
	# (SC2009 recommends using pgrep, but it is not always available)
	dep_manager=$(ps aux | grep dep_manager | grep -v grep)
	if [ -n "$dep_manager" ]; then
		echo "DEP is already running"
		exit 0
	fi

	# Some basic checks before proceeding
	if [ ! -f "$ACTIVE_IMAGE_ID_PATH" ]; then
		echo "$ACTIVE_IMAGE_ID_PATH not found, can't select active rootfs"
		exit 1
	fi

	active_image_id=$(cat "$ACTIVE_IMAGE_ID_PATH")
	rootfs="$IMAGES_PATH/$active_image_id/rootfs_squash"

	if [ ! -f "$rootfs" ]; then
		echo "$rootfs not found"
		exit 1
	fi

	mkdir -p /var/run/dante
	mkdir -p ${CONTAINER_STATUS_PATH}

	# Make sure /etc/resolv.conf is there when later on we
	# try to bind mount it from the container.
	if [ ! -f "/etc/resolv.conf" ]; then
		touch /etc/resolv.conf
	fi

	if ! grep -q " $ROOTFS_MOUNTPOINT " /proc/mounts; then
		if ! mount "$rootfs" "$ROOTFS_MOUNTPOINT" >/dev/null 2>&1; then
			echo "could not mount $rootfs"
			exit 1
		fi
	fi

	container_logfile="/var/log/dante_container.log"
	if ! ${CONTAINER_CMD} run --detach --bundle $DEP_EXE_PATH/bundle dante > "$container_logfile" 2>&1; then
		echo "failed to start dante container, check $container_logfile"
	else
		echo "DEP started"
	fi
}

stop()
{
	# in some cases we might have the mountpoint but no container running:
	# check if that's the case before proceeding
	if ${CONTAINER_CMD} list | grep dante >/dev/null 2>&1; then
		# stop the init process (dep_manager) by sending a SIGTERM signal
		echo "stopping DEP..."
		${CONTAINER_CMD} kill dante TERM

		for _ in $(seq 1 10); do
			sleep 1
			DEP_PROCS=$(${CONTAINER_CMD} ps dante | grep -v PID -c)
			if [ "$DEP_PROCS" -eq 0 ]; then
				break
			fi
		done

		DEP_PROCS=$(${CONTAINER_CMD} ps dante | grep -v PID -c)
		if [ "$DEP_PROCS" -ne 0 ]; then
			echo "DEP still running, sending SIGKILL"
			${CONTAINER_CMD} kill -a dante KILL
			sleep 1
		fi

		echo "remove container..."
		${CONTAINER_CMD} delete --force dante
	fi

	if grep -q " $ROOTFS_MOUNTPOINT " /proc/mounts; then
		echo "umount rootfs..."
		umount $DEP_EXE_PATH/bundle/rootfs
	fi

	echo "done"
}

USAGE_MESSAGE="Usage: dep.sh <start|stop>"

if [ "$#" -eq 0 ]; then
	echo "$USAGE_MESSAGE"
	exit 1
fi

case $1 in
	"start" ) start "$2" ;;
	"stop" ) stop ;;
	* )
		echo "$USAGE_MESSAGE"
		exit 1
		;;
esac
